package com.androidbook.simpleservice;

import java.util.Date;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.location.Criteria;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;
import android.os.IBinder;
import android.util.Log;
import android.widget.Toast;

public class GPXService extends Service {
    private static final int GPS_NOTIFY = 0x2001;
    private static final String DEBUG_TAG = "GPXService";
    public static final String EXTRA_UPDATE_RATE = "update-rate";
    public static final String GPX_SERVICE = "com.androidbook.GPXService.SERVICE";
    private LocationManager location = null;
    private NotificationManager notifier = null;
    private int updateRate = -1;

    @Override
    public void onCreate() {
        super.onCreate();
        location = (LocationManager) getSystemService(Context.LOCATION_SERVICE);
        notifier =
            (NotificationManager) getSystemService(Context.NOTIFICATION_SERVICE);
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        // Android 2.0, poziom 5 
        Log.v(DEBUG_TAG, "Wywołano metodę onStartCommand(), aplikacja musi działać w SDK poziomu 5. lub nowszym");
        if (flags != 0) {
            Log.w(DEBUG_TAG, "Ponowna próba startu usługi: " + flags);
        }
        doServiceStart(intent, startId);
        return Service.START_REDELIVER_INTENT;
    }

    @Override
    public void onStart(Intent intent, int startId) {
        // wersje wcześniejsze niż Android 2.0
        super.onStart(intent, startId);
        Log.v(DEBUG_TAG, "Wywołano metodę onStart(), aplikacja musi działać w SDK poziomu 3. lub 4.");
        doServiceStart(intent, startId);
    }

    private void doServiceStart(Intent intent, int startId) {
        updateRate = intent.getIntExtra(EXTRA_UPDATE_RATE, -1);
        if (updateRate == -1) {
            updateRate = 60000;
        }
        Criteria criteria = new Criteria();
        criteria.setAccuracy(Criteria.NO_REQUIREMENT);
        criteria.setPowerRequirement(Criteria.POWER_LOW);
        location = (LocationManager) getSystemService(Context.LOCATION_SERVICE);
        String best = location.getBestProvider(criteria, true);
        location.requestLocationUpdates(best, updateRate, 0, trackListener);
        // notify that we've started up
        Notification notify =
            new Notification(android.R.drawable.stat_notify_more, "Śledzenie odczytu GPS",
                System.currentTimeMillis());
        notify.flags |= Notification.FLAG_AUTO_CANCEL;
        Intent toLaunch =
            new Intent(getApplicationContext(), ServiceControlActivity.class);
        PendingIntent intentBack =
            PendingIntent.getActivity(getApplicationContext(), 0, toLaunch, 0);
        notify.setLatestEventInfo(getApplicationContext(), "Śledzenie sygnału GPS",
            "Śledzenie odbywa się w odstępach " + updateRate + " ms, przy czym używany jest dostawca [" + best
                + "].", intentBack);
        notifier.notify(GPS_NOTIFY, notify);
    }

    @Override
    public void onDestroy() {
        Log.v(DEBUG_TAG, "onDestroy() called");
        if (location != null) {
            location.removeUpdates(trackListener);
            location = null;
        }
        // informacja o zakończeniu śledzenia
        Notification notify =
            new Notification(android.R.drawable.stat_notify_more, "Śledzenie odczytu GPS",
                System.currentTimeMillis());
        notify.flags |= Notification.FLAG_AUTO_CANCEL;
        Intent toLaunch =
            new Intent(getApplicationContext(), ServiceControlActivity.class);
        PendingIntent intentBack =
            PendingIntent.getActivity(getApplicationContext(), 0, toLaunch, 0);
        notify.setLatestEventInfo(getApplicationContext(), "Śledzenie odczytu GPS",
            "Śledzenie zatrzymane", intentBack);
        notifier.notify(GPS_NOTIFY, notify);
        super.onDestroy();
    }

    @Override
    public IBinder onBind(Intent intent) {
        // Mamy tylko jedną intencję, więc nie trzeba jej sprawdzać
        return mRemoteInterfaceBinder;
    }

    // nasz zdalny interfejs
    private final IRemoteInterface.Stub mRemoteInterfaceBinder =
        new IRemoteInterface.Stub() {
            public Location getLastLocation() {
                Log.v("interface", "Wywołano metdoę getLastLocation().");
                return lastLocation;
            }

            public GPXPoint getGPXPoint() {
                if (lastLocation == null) {
                    return null;
                } else {
                    Log.v("interface", "Wywołano metodę getGPXPoint()");
                    GPXPoint point = new GPXPoint();
                    point.elevation = lastLocation.getAltitude();
                    point.latitude = (int) (lastLocation.getLatitude() * 1E6);
                    point.longitude = (int) (lastLocation.getLongitude() * 1E6);
                    point.timestamp = new Date(lastLocation.getTime());
                    return point;
                }
            }
        };
    private Location firstLocation = null;
    private Location lastLocation = null;
    private long lastTime = -1;
    private long firstTime = -1;
    private LocationListener trackListener = new LocationListener() {
        public void onLocationChanged(Location location) {
            long thisTime = System.currentTimeMillis();
            long diffTime = thisTime - lastTime;
            Log.v(DEBUG_TAG, "diffTime == " + diffTime + ", updateRate = "
                + updateRate);
            if (diffTime < updateRate) {
                // jeszcze nie upłynęło dostatecznie dużo czasu
                return;
            }
            lastTime = thisTime;
            String locInfo =
                String.format("Aktualne położenie = (%f, %f) na wysokości: %.1f metrów.",
                    location.getLatitude(), location.getLongitude(), location
                        .getAltitude());
            if (lastLocation != null) {
                float distance = location.distanceTo(lastLocation);
                locInfo +=
                    String
                        .format("\n Odległość od poprzedniego = %.1f metrów", distance);
                float lastSpeed = distance / diffTime;
                locInfo += String.format("\n\tPrędkość: %.1fm/s", lastSpeed);
                if (location.hasSpeed()) {
                    float gpsSpeed = location.getSpeed();
                    locInfo +=
                        String.format(" (lub %.1fm/s)", lastSpeed, gpsSpeed);
                } else {
                }
            }
            if (firstLocation != null && firstTime != -1) {
                float overallDistance = location.distanceTo(firstLocation);
                float overallSpeed = overallDistance / (thisTime - firstTime);
                locInfo +=
                    String.format("\n\tPrędkość ogólna: %.1fm/s na %.1f metrach",
                        overallSpeed, overallDistance);
            }
            lastLocation = location;
            if (firstLocation == null) {
                firstLocation = location;
                firstTime = thisTime;
            }
            Toast.makeText(getApplicationContext(), locInfo, Toast.LENGTH_LONG)
                .show();
            Log.v(DEBUG_TAG, "Testy");
        }

        public void onProviderDisabled(String provider) {
            Log.v(DEBUG_TAG, "Dostawca wyłączony " + provider);
        }

        public void onProviderEnabled(String provider) {
            Log.v(DEBUG_TAG, "Dostawca włączony " + provider);
        }

        public void onStatusChanged(String provider, int status, Bundle extras) {
        }
    };
}
